<?php
if (!defined("WHMCS")) {
    die("This file cannot be accessed directly.");
}

function mhpay_MetaData()
{
    return [
        'DisplayName' => 'MhPay Payment Gateway',
        'APIVersion' => '1.1',
        'DisableLocalCreditCardInput' => true,
        'TokenisedStorage' => false,
    ];
}

function mhpay_config()
{
    return [
        'FriendlyName' => [
            'Type' => 'System',
            'Value' => 'mhpay',
        ],
        'Description' => [
            'Type' => 'System',
            'Value' => 'This is the mhpay payment gateway module for WHMCS, which allows users to process payments securely.',
        ],
        'DeveloperName' => [
            'Type' => 'System',
            'Value' => 'Mouzh Hossain Shad',
        ],
        'Version' => [
            'Type' => 'System',
            'Value' => '1.0.3',
        ],
        'apikey' => [
            'FriendlyName' => 'API Key',
            'Type' => 'password',
            'Size' => '64',
        ],
        'baseUrl' => [
            'FriendlyName' => 'Base URL (no trailing slash)',
            'Type' => 'text',
            'Default' => 'https://sandbox.mhpay.cc',
        ],
        'alternateUrl' => [
            'FriendlyName' => 'Alternate API URL (if main fails)',
            'Type' => 'text',
            'Default' => 'https://mhpay.cc',
            'Description' => 'Try production URL if sandbox returns 526 error',
        ],
        'returnType' => [
            'FriendlyName' => 'Return Type',
            'Type' => 'dropdown',
            'Options' => 'GET,POST',
            'Default' => 'POST',
        ],
        'currency_pp' => [
            'FriendlyName' => 'Currency',
            'Type' => 'text',
            'Default' => 'BDT',
        ],
        'debug' => [
            'FriendlyName' => 'Debug Mode',
            'Type' => 'yesno',
            'Description' => 'Enable to see detailed error messages',
        ],
        'useAlternate' => [
            'FriendlyName' => 'Use Alternate URL',
            'Type' => 'yesno',
            'Description' => 'Check this if main URL returns 526 error',
        ],
    ];
}

function mhpay_link($params)
{
    $apiKey = $params['apikey'];
    
    // Choose URL based on settings
    $baseUrl = $params['useAlternate'] 
        ? rtrim($params['alternateUrl'], '/') 
        : rtrim($params['baseUrl'], '/');
    
    $invoiceId = $params['invoiceid'];
    $amount = $params['amount'];
    $currency = $params['currency_pp'];
    $returnUrl = $params['returnurl'];
    $debugMode = $params['debug'] ?? false;
    
    // Build callback URL
    $parsedUrl = parse_url($params['systemurl']);
    $scheme = isset($parsedUrl['scheme']) ? $parsedUrl['scheme'] : 'http';
    $host = $parsedUrl['host'] ?? '';
    $port = isset($parsedUrl['port']) ? ':' . $parsedUrl['port'] : '';
    
    $systemUrl = $scheme . '://' . $host . $port;
    $callbackUrl = $systemUrl . '/modules/gateways/callback/mhpay.php';
    
    // Prepare API request
    $postData = [
        'full_name'    => $params['clientdetails']['fullname'],
        'email_mobile' => $params['clientdetails']['email'],
        'amount'       => $amount,
        'metadata'     => ['invoiceid' => $invoiceId],
        'redirect_url' => $returnUrl,
        'cancel_url'   => $params['systemurl'],
        'webhook_url'  => $callbackUrl,
        'return_type'  => $params['returnType'],
        'currency'     => $currency,
    ];
    
    // Try primary endpoint
    $apiUrl = $baseUrl . '/api/create-charge';
    $result = mhpay_make_request($apiUrl, $apiKey, $postData, $debugMode, $invoiceId);
    
    // If 526 error and not using alternate, try alternate URL
    if ($result['http_code'] == 526 && !$params['useAlternate'] && !empty($params['alternateUrl'])) {
        logActivity('mhpay: Got 526 error, trying alternate URL');
        $alternateUrl = rtrim($params['alternateUrl'], '/') . '/api/create-charge';
        $result = mhpay_make_request($alternateUrl, $apiKey, $postData, $debugMode, $invoiceId);
    }
    
    // Check for successful response
    if (isset($result['data']['pp_url']) && !empty($result['data']['pp_url'])) {
        return '<a href="' . htmlspecialchars($result['data']['pp_url']) . '" target="_blank" class="btn btn-primary">Pay Now</a>';
    }
    
    // Handle error response
    $errorMsg = $result['error'] ?? 'Unknown error';
    
    if ($debugMode) {
        return '<div class="alert alert-danger">
            <strong>mhpay Error:</strong><br>
            ' . htmlspecialchars($errorMsg) . '<br><br>
            <small>HTTP Code: ' . $result['http_code'] . '</small><br>
            <small>URL: ' . htmlspecialchars($result['url']) . '</small><br>
            <small>Response: ' . htmlspecialchars(substr($result['raw_response'], 0, 500)) . '</small>
        </div>';
    }
    
    return '<div class="alert alert-danger">Error processing payment request. Please contact support. (Error: HTTP ' . $result['http_code'] . ')</div>';
}

function mhpay_make_request($apiUrl, $apiKey, $postData, $debugMode, $invoiceId)
{
    // Initialize cURL
    $ch = curl_init($apiUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'accept: application/json',
        'mh-mhpay-api-key: ' . $apiKey,
    ]);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($postData));
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    // Force TLS 1.2 or higher
    if (defined('CURL_SSLVERSION_TLSv1_2')) {
        curl_setopt($ch, CURLOPT_SSLVERSION, CURL_SSLVERSION_TLSv1_2);
    }
    
    // SSL Configuration
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
    
    // Try to locate CA bundle
    $caBundlePaths = [
        '/etc/ssl/certs/ca-certificates.crt',
        '/etc/pki/tls/certs/ca-bundle.crt',
        '/usr/share/ssl/certs/ca-bundle.crt',
        '/usr/local/share/certs/ca-root-nss.crt',
        '/etc/ssl/cert.pem',
    ];
    
    foreach ($caBundlePaths as $path) {
        if (file_exists($path)) {
            curl_setopt($ch, CURLOPT_CAINFO, $path);
            break;
        }
    }
    
    // Execute request
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);
    
    // Always log the attempt
    logActivity('mhpay API Call - Invoice #' . $invoiceId);
    logActivity('mhpay URL: ' . $apiUrl);
    logActivity('mhpay HTTP Code: ' . $httpCode);
    
    // Handle cURL errors
    if ($response === false) {
        $errorMsg = 'cURL Error: ' . $curlError;
        logActivity('mhpay cURL Error: ' . $errorMsg);
        
        return [
            'success' => false,
            'http_code' => 0,
            'error' => $errorMsg,
            'raw_response' => '',
            'data' => null,
            'url' => $apiUrl
        ];
    }
    
    // Log response
    logActivity('mhpay Raw Response: ' . substr($response, 0, 500));
    
    // Decode response
    $result = json_decode($response, true);
    
    // Check for JSON errors
    if (json_last_error() !== JSON_ERROR_NONE) {
        logActivity('mhpay JSON Decode Error: ' . json_last_error_msg());
        $errorMsg = 'Invalid response format (HTTP ' . $httpCode . ')';
    } else {
        $errorMsg = 'HTTP ' . $httpCode;
        if (isset($result['message'])) {
            $errorMsg .= ': ' . $result['message'];
        } elseif (isset($result['error'])) {
            $errorMsg .= ': ' . $result['error'];
        }
    }
    
    return [
        'success' => ($httpCode == 200 || $httpCode == 201) && isset($result['pp_url']),
        'http_code' => $httpCode,
        'error' => $errorMsg,
        'raw_response' => $response,
        'data' => $result,
        'url' => $apiUrl
    ];
}